<?php
/**
 * Stripe PaymentIntent
 *
 * @package SimplePay\Core\Payments\PaymentIntent
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.6.0
 */

namespace SimplePay\Core\Payments\PaymentIntent;

use SimplePay\Core\Payments\Stripe_API;
use SimplePay\Core\Legacy;

use function SimplePay\Core\SimplePay;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Retrieves a PaymentIntent.
 *
 * @since 3.8.0
 *
 * @param string|array $payment_intent Payment Intent ID or {
 *   Arguments used to retrieve a PaymentIntent.
 *
 *   @type string $id Payment Intent ID.
 * }
 * @param array        $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \Stripe\PaymentIntent
 */
function retrieve( $payment_intent, $api_request_args = array() ) {
	if ( false === is_array( $payment_intent ) ) {
		$payment_intent_args = array(
			'id' => $payment_intent,
		);
	} else {
		$payment_intent_args = $payment_intent;
	}

	return Stripe_API::request(
		'PaymentIntent',
		'retrieve',
		$payment_intent_args,
		$api_request_args
	);
}

/**
 * Retrieves PaymentIntents.
 *
 * @since 3.8.0
 *
 * @param array $payment_intents Optional arguments used when listing PaymentIntents.
 * @param array $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return object
 */
function all( $payment_intents = array(), $api_request_args = array() ) {
	return Stripe_API::request(
		'PaymentIntent',
		'all',
		$payment_intents,
		$api_request_args
	);
}

/**
 * Creates a PaymentIntent.
 *
 * @since 3.6.0
 *
 * @param array $paymentintent_args Arguments used to create a PaymentIntent.
 * @param array $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \Stripe\PaymentIntent
 */
function create( $paymentintent_args, $api_request_args = array() ) {
	$defaults           = array();
	$paymentintent_args = wp_parse_args( $paymentintent_args, $defaults );

	/**
	 * Filter the arguments used to generate a PaymentIntent.
	 *
	 * @since 3.6.0
	 *
	 * @param array $payment_intent_args Arguemnts used to generate a PaymentIntent.
	 */
	$paymentintent_args = apply_filters( 'simpay_create_paymentintent_args', $paymentintent_args );

	/**
	 * Allows processing before a PaymentIntenet is created.
	 *
	 * @since 3.6.0
	 *
	 * @param array $paymentintent_args Arguments used to create a PaymentIntent.
	 */
	do_action( 'simpay_before_paymentintent_created', $paymentintent_args );

	// Create PaymentIntent.
	$paymentintent = Stripe_API::request(
		'PaymentIntent',
		'create',
		$paymentintent_args,
		$api_request_args
	);

	/**
	 * Allows further processing after a PaymentIntent has been created.
	 *
	 * @since 3.6.0
	 *
	 * @param \Stripe\PaymentIntent $paymentintent PaymentIntent.
	 */
	do_action( 'simpay_after_paymentintent_created', $paymentintent );

	return $paymentintent;
}

/**
 * Confirms a PaymentIntent.
 *
 * @since 3.6.0
 *
 * @param string $paymentintent_id PaymentIntent ID to confirm.
 * @param array  $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \Stripe\PaymentIntent
 */
function confirm( $paymentintent_id, $api_request_args = array() ) {
	$paymentintent = Stripe_API::request(
		'PaymentIntent',
		'retrieve',
		$paymentintent_id,
		$api_request_args
	);

	$paymentintent->confirm();

	return $paymentintent;
}

/**
 * Uses the current payment form request to generate arguments for a PaymentIntent.
 *
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @param int                           $customer_id Stripe Customer ID.
 * @return array
 */
function get_args_from_payment_form_request( $form, $form_data, $form_values, $customer_id ) {
	$amount = isset( $form_values['simpay_amount'] )
		? $form_values['simpay_amount']
		: simpay_convert_amount_to_cents( $form->amount );

	$paymentintent_args = array(
		'amount'      => $amount,
		'currency'    => $form->currency,
		'metadata'    => array_merge(
			array(
				'simpay_form_id' => $form->id,
			),
			Legacy\Hooks\simpay_payment_metadata( $form, $form_data, $form_values, $customer_id )
		),
		'description' => Legacy\Hooks\simpay_payment_description( $form, $form_data, $form_values, $customer_id ),
	);

	if ( ! empty( $form->statement_descriptor ) ) {
		$paymentintent_args['statement_descriptor'] = $form->statement_descriptor;
	}

	// Legacy filter.
	$paymentintent_args = Legacy\Hooks\simpay_stripe_charge_args( $paymentintent_args, $form, $form_values );

	/**
	 * Filters arguments used to create a PaymentIntent from a payment form request.
	 *
	 * @since 3.6.0
	 *
	 * @param array                         $paymentintent_args Arguments for PaymentIntent.
	 * @param SimplePay\Core\Abstracts\Form $form Form instance.
	 * @param array                         $form_data Form data generated by the client.
	 * @param array                         $form_values Values of named fields in the payment form.
	 * @param int                           $customer_id Stripe Customer ID.
	 * @return array
	 */
	$paymentintent_args = apply_filters(
		'simpay_get_paymentintent_args_from_payment_form_request',
		$paymentintent_args,
		$form,
		$form_data,
		$form_values,
		$customer_id
	);

	return $paymentintent_args;
}
